import sqlite3
from flask import Flask, render_template, request, redirect, url_for, g

app = Flask(__name__)
app.config['SECRET_KEY'] = 'your_secret_key'
DATABASE = 'database.db'

# --- Database Setup ---
def get_db():
    db = getattr(g, '_database', None)
    if db is None:
        db = g._database = sqlite3.connect(DATABASE)
        db.row_factory = sqlite3.Row
    return db

@app.teardown_appcontext
def close_connection(exception):
    db = getattr(g, '_database', None)
    if db is not None:
        db.close()

def init_db():
    with app.app_context():
        db = get_db()
        # Create Questions Table
        db.execute('''
            CREATE TABLE IF NOT EXISTS questions (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                question_text TEXT NOT NULL,
                option_a TEXT NOT NULL,
                option_b TEXT NOT NULL,
                option_c TEXT NOT NULL,
                option_d TEXT NOT NULL,
                correct_answer TEXT NOT NULL
            )
        ''')
        db.commit()

# --- Routes ---

@app.route('/')
def index():
    return render_template('index.html')

@app.route('/admin', methods=['GET', 'POST'])
def admin():
    """Question Bank Management"""
    if request.method == 'POST':
        question = request.form['question']
        op_a = request.form['option_a']
        op_b = request.form['option_b']
        op_c = request.form['option_c']
        op_d = request.form['option_d']
        correct = request.form['correct_answer']
        
        db = get_db()
        db.execute('INSERT INTO questions (question_text, option_a, option_b, option_c, option_d, correct_answer) VALUES (?, ?, ?, ?, ?, ?)',
                   (question, op_a, op_b, op_c, op_d, correct))
        db.commit()
        return redirect(url_for('admin'))
    
    # Fetch all questions to display
    db = get_db()
    questions = db.execute('SELECT * FROM questions').fetchall()
    return render_template('admin.html', questions=questions)

@app.route('/quiz')
def quiz():
    """The Timed Quiz Interface"""
    db = get_db()
    # Fetch 5 random questions
    questions = db.execute('SELECT * FROM questions ORDER BY RANDOM() LIMIT 5').fetchall()
    return render_template('quiz.html', questions=questions)

@app.route('/submit', methods=['POST'])
def submit():
    """Real-time Scoring Logic"""
    db = get_db()
    score = 0
    total = 0
    results = []

    for key, value in request.form.items():
        if key.startswith('q_'):
            q_id = int(key.split('_')[1])
            user_answer = value
            
            # Verify answer
            question = db.execute('SELECT * FROM questions WHERE id = ?', (q_id,)).fetchone()
            if question:
                total += 1
                is_correct = (user_answer == question['correct_answer'])
                if is_correct:
                    score += 1
                
                results.append({
                    'question': question['question_text'],
                    'your_answer': user_answer,
                    'correct_answer': question['correct_answer'],
                    'is_correct': is_correct
                })

    return render_template('result.html', score=score, total=total, results=results)

if __name__ == '__main__':
    init_db() # Initialize DB on first run
    app.run(debug=True)